"""
graph_utils.py

Utilities for boundary‑graph operations within the tick‑flip algebra.

Provides functions to load and validate the boundary subgraph representations used
in context‑graph simulations.
"""

import numpy as np


def load_boundary_graph(path: str) -> np.ndarray:
    """
    Load the adjacency matrix of a boundary subgraph from a .npy file.

    Parameters
    ----------
    path : str
        Path to the .npy file containing a square adjacency matrix.

    Returns
    -------
    np.ndarray
        The loaded adjacency matrix.
    """
    graph = np.load(path, allow_pickle=True)
    return graph


def validate_graph(graph: np.ndarray) -> bool:
    """
    Validate the boundary graph adjacency matrix.

    Checks:
      - graph is a NumPy ndarray
      - graph is two-dimensional and square
      - graph is symmetric (undirected)

    Parameters
    ----------
    graph : np.ndarray
        Adjacency matrix to validate.

    Returns
    -------
    bool
        True if validation succeeds.

    Raises
    ------
    TypeError
        If graph is not a numpy.ndarray.
    ValueError
        If graph is not square or not symmetric.
    """
    if not isinstance(graph, np.ndarray):
        raise TypeError(f"Expected numpy.ndarray, got {type(graph)}")
    if graph.ndim != 2 or graph.shape[0] != graph.shape[1]:
        raise ValueError(f"Expected square matrix, got shape {graph.shape}")
    if not np.allclose(graph, graph.T):
        raise ValueError("Adjacency matrix must be symmetric")
    return True
